<?php
// Webhook para receber notificações da API Moonfy
// Baseado na documentação: https://moonfy.readme.io/reference/formato-dos-postbacks

header('Content-Type: application/json');

// Log de todas as requisições recebidas
function logWebhook($data) {
    $logFile = 'webhook-moonfy.log';
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] " . json_encode($data) . "\n";
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}

// Função para processar webhook de transação
function processTransactionWebhook($data) {
    $transaction = $data['data'] ?? null;
    
    if (!$transaction) {
        return ['success' => false, 'error' => 'Dados da transação não encontrados'];
    }

    $transactionId = $transaction['id'] ?? null;
    $status = $transaction['status'] ?? null;
    $amount = $transaction['amount'] ?? 0;
    $customer = $transaction['customer'] ?? null;
    $pix = $transaction['pix'] ?? null;

    // Log da transação
    logWebhook([
        'type' => 'transaction_webhook',
        'transaction_id' => $transactionId,
        'status' => $status,
        'amount' => $amount,
        'customer_name' => $customer['name'] ?? 'N/A',
        'customer_cpf' => $customer['document']['number'] ?? 'N/A'
    ]);

    // Processar diferentes status
    switch ($status) {
        case 'paid':
            // Pagamento aprovado
            processPaidTransaction($transaction);
            break;
            
        case 'refused':
            // Pagamento recusado
            processRefusedTransaction($transaction);
            break;
            
        case 'canceled':
            // Transação cancelada
            processCanceledTransaction($transaction);
            break;
            
        case 'expired':
            // Transação expirada
            processExpiredTransaction($transaction);
            break;
            
        default:
            // Outros status
            logWebhook([
                'type' => 'unknown_status',
                'transaction_id' => $transactionId,
                'status' => $status
            ]);
    }

    return ['success' => true, 'processed' => true];
}

// Processar transação paga
function processPaidTransaction($transaction) {
    $transactionId = $transaction['id'];
    $amount = $transaction['amount'] / 100; // Converter de centavos para reais
    $customer = $transaction['customer'];
    $paidAt = $transaction['paidAt'] ?? date('Y-m-d H:i:s');

    logWebhook([
        'type' => 'payment_confirmed',
        'transaction_id' => $transactionId,
        'amount' => $amount,
        'customer_name' => $customer['name'] ?? 'N/A',
        'paid_at' => $paidAt
    ]);

    // Aqui você pode adicionar lógica adicional como:
    // - Enviar email de confirmação
    // - Atualizar banco de dados
    // - Notificar sistema de entrega
    // - Etc.
}

// Processar transação recusada
function processRefusedTransaction($transaction) {
    $transactionId = $transaction['id'];
    $refusedReason = $transaction['refusedReason'] ?? null;

    logWebhook([
        'type' => 'payment_refused',
        'transaction_id' => $transactionId,
        'reason' => $refusedReason
    ]);
}

// Processar transação cancelada
function processCanceledTransaction($transaction) {
    $transactionId = $transaction['id'];

    logWebhook([
        'type' => 'payment_canceled',
        'transaction_id' => $transactionId
    ]);
}

// Processar transação expirada
function processExpiredTransaction($transaction) {
    $transactionId = $transaction['id'];

    logWebhook([
        'type' => 'payment_expired',
        'transaction_id' => $transactionId
    ]);
}

// Função para processar webhook de checkout
function processCheckoutWebhook($data) {
    $checkout = $data['data'] ?? null;
    
    if (!$checkout) {
        return ['success' => false, 'error' => 'Dados do checkout não encontrados'];
    }

    $checkoutId = $checkout['id'] ?? null;
    $transaction = $checkout['transaction'] ?? null;

    logWebhook([
        'type' => 'checkout_webhook',
        'checkout_id' => $checkoutId,
        'has_transaction' => !empty($transaction)
    ]);

    // Se há uma transação associada, processar ela também
    if ($transaction) {
        return processTransactionWebhook(['data' => $transaction]);
    }

    return ['success' => true, 'processed' => true];
}

// Função para processar webhook de transferência
function processTransferWebhook($data) {
    $transfer = $data['data'] ?? null;
    
    if (!$transfer) {
        return ['success' => false, 'error' => 'Dados da transferência não encontrados'];
    }

    $transferId = $transfer['id'] ?? null;
    $status = $transfer['status'] ?? null;
    $amount = $transfer['amount'] ?? 0;

    logWebhook([
        'type' => 'transfer_webhook',
        'transfer_id' => $transferId,
        'status' => $status,
        'amount' => $amount
    ]);

    return ['success' => true, 'processed' => true];
}

// Processar requisição POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Obter dados do webhook
        $input = file_get_contents('php://input');
        $data = json_decode($input, true);

        if (!$data) {
            throw new Exception('Dados JSON inválidos');
        }

        // Log da requisição recebida
        logWebhook([
            'type' => 'webhook_received',
            'raw_data' => $data
        ]);

        $type = $data['type'] ?? null;
        $objectId = $data['objectId'] ?? null;

        // Processar baseado no tipo
        switch ($type) {
            case 'transaction':
                $result = processTransactionWebhook($data);
                break;
                
            case 'checkout':
                $result = processCheckoutWebhook($data);
                break;
                
            case 'transfer':
                $result = processTransferWebhook($data);
                break;
                
            default:
                $result = [
                    'success' => false, 
                    'error' => "Tipo de webhook não reconhecido: $type"
                ];
        }

        // Responder com sucesso
        http_response_code(200);
        echo json_encode($result);

    } catch (Exception $e) {
        // Log do erro
        logWebhook([
            'type' => 'webhook_error',
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);

        // Responder com erro
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
} else {
    // Método não permitido
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'error' => 'Método não permitido'
    ]);
}
?>
